"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebCryptoRawAesMaterial = exports.NodeRawAesMaterial = void 0;
/* Here I am reusing the Material implementation and interface from material-management.
 * This is because there are many security guarantees that this implementations offer
 * that map to the current implementation of raw AES keyrings.
 * The KeyringTrace is an unfortunate case because there is no mapping.
 * However the trade off seems worth it and the convolutions to make it work seem minimal.
 */
const material_management_1 = require("@aws-crypto/material-management");
const raw_aes_algorithm_suite_1 = require("./raw_aes_algorithm_suite");
class NodeRawAesMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptionContext = Object.freeze({});
    constructor(suiteId) {
        /* Precondition: NodeRawAesMaterial suiteId must be RawAesWrappingSuiteIdentifier. */
        (0, material_management_1.needs)(raw_aes_algorithm_suite_1.RawAesWrappingSuiteIdentifier[suiteId], 'suiteId not supported.');
        this.suite = new material_management_1.NodeAlgorithmSuite(suiteId);
        /* NodeRawAesMaterial need to set a flag, this is an abuse of TraceFlags
         * because the material is not generated.
         * but CryptographicMaterial force a flag to be set.
         */
        const setFlags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        (0, material_management_1.decorateCryptographicMaterial)(this, setFlags);
        Object.setPrototypeOf(this, NodeRawAesMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
exports.NodeRawAesMaterial = NodeRawAesMaterial;
(0, material_management_1.frozenClass)(NodeRawAesMaterial);
class WebCryptoRawAesMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext = Object.freeze({});
    constructor(suiteId) {
        /* Precondition: WebCryptoAlgorithmSuite suiteId must be RawAesWrappingSuiteIdentifier. */
        (0, material_management_1.needs)(raw_aes_algorithm_suite_1.RawAesWrappingSuiteIdentifier[suiteId], 'suiteId not supported.');
        this.suite = new material_management_1.WebCryptoAlgorithmSuite(suiteId);
        this.validUsages = Object.freeze([
            'decrypt',
            'encrypt',
        ]);
        /* WebCryptoRawAesMaterial need to set a flag, this is an abuse of TraceFlags
         * because the material is not generated.
         * but CryptographicMaterial force a flag to be set.
         */
        const setFlag = material_management_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        (0, material_management_1.decorateCryptographicMaterial)(this, setFlag);
        (0, material_management_1.decorateWebCryptoMaterial)(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoRawAesMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey && this.hasCryptoKey;
    }
}
exports.WebCryptoRawAesMaterial = WebCryptoRawAesMaterial;
(0, material_management_1.frozenClass)(WebCryptoRawAesMaterial);
//# sourceMappingURL=data:application/json;base64,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